<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;

/**
 * Command BackupDatabase
 * 
 * Membuat backup database terenkripsi untuk disaster recovery
 * Dapat dijalankan manual atau dijadwalkan
 */
class BackupDatabase extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'backup:database 
                            {--compress : Compress the backup file}
                            {--encrypt : Encrypt the backup file}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Backup database dengan enkripsi untuk keamanan data sensitif';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting database backup...');

        // Get database configuration
        $database = config('database.connections.' . config('database.default'));
        
        if ($database['driver'] !== 'mysql') {
            $this->error('This command currently only supports MySQL databases.');
            return 1;
        }

        // Create backup filename dengan timestamp
        $timestamp = now()->format('Y-m-d_His');
        $filename = "backup_{$database['database']}_{$timestamp}.sql";
        $backupPath = storage_path("app/backups");

        // Buat direktori backup jika belum ada
        if (!file_exists($backupPath)) {
            mkdir($backupPath, 0755, true);
        }

        $fullPath = "{$backupPath}/{$filename}";

        // Build mysqldump command
        $command = sprintf(
            'mysqldump --user=%s --password=%s --host=%s --port=%s %s > %s',
            escapeshellarg($database['username']),
            escapeshellarg($database['password']),
            escapeshellarg($database['host']),
            escapeshellarg($database['port']),
            escapeshellarg($database['database']),
            escapeshellarg($fullPath)
        );

        // Jalankan backup
        $this->info('Creating database dump...');
        
        $process = Process::fromShellCommandline($command);
        $process->setTimeout(3600); // 1 hour timeout
        
        try {
            $process->mustRun();
            $this->info("✓ Database backup created: {$filename}");
        } catch (ProcessFailedException $e) {
            $this->error('Backup failed: ' . $e->getMessage());
            return 1;
        }

        // Compress jika diminta
        if ($this->option('compress')) {
            $this->info('Compressing backup file...');
            $compressedPath = $fullPath . '.gz';
            
            $compressCommand = sprintf('gzip %s', escapeshellarg($fullPath));
            $compressProcess = Process::fromShellCommandline($compressCommand);
            
            try {
                $compressProcess->mustRun();
                $this->info('✓ Backup compressed');
                $fullPath = $compressedPath;
                $filename = $filename . '.gz';
            } catch (ProcessFailedException $e) {
                $this->warn('Compression failed, keeping uncompressed backup');
            }
        }

        // Enkripsi jika diminta (menggunakan Laravel encryption)
        if ($this->option('encrypt')) {
            $this->info('Encrypting backup file...');
            
            try {
                $content = file_get_contents($fullPath);
                $encrypted = encrypt($content);
                $encryptedPath = $fullPath . '.enc';
                
                file_put_contents($encryptedPath, $encrypted);
                
                // Hapus file unencrypted
                unlink($fullPath);
                
                $this->info('✓ Backup encrypted');
                $filename = $filename . '.enc';
            } catch (\Exception $e) {
                $this->error('Encryption failed: ' . $e->getMessage());
                return 1;
            }
        }

        // Cleanup old backups (keep only last 30 days)
        $this->cleanupOldBackups($backupPath);

        // Display file size
        $filesize = filesize("{$backupPath}/{$filename}");
        $filesizeFormatted = $this->formatBytes($filesize);
        
        $this->info("✓ Backup completed successfully!");
        $this->info("File: {$filename}");
        $this->info("Size: {$filesizeFormatted}");
        $this->info("Location: {$backupPath}");
        
        // Security reminder
        $this->warn('');
        $this->warn('SECURITY REMINDER:');
        $this->warn('- Store backup di lokasi yang aman');
        $this->warn('- Backup terenkripsi memerlukan APP_KEY untuk restore');
        $this->warn('- Test restore procedure secara berkala');
        $this->warn('- Simpan backup offline untuk disaster recovery');

        return 0;
    }

    /**
     * Cleanup old backup files
     */
    protected function cleanupOldBackups($path)
    {
        $this->info('Cleaning up old backups...');
        
        $files = glob($path . '/backup_*.{sql,sql.gz,sql.enc,sql.gz.enc}', GLOB_BRACE);
        $cutoffDate = now()->subDays(30);
        $deletedCount = 0;

        foreach ($files as $file) {
            if (filemtime($file) < $cutoffDate->timestamp) {
                unlink($file);
                $deletedCount++;
            }
        }

        if ($deletedCount > 0) {
            $this->info("✓ Deleted {$deletedCount} old backup(s)");
        }
    }

    /**
     * Format bytes ke human readable
     */
    protected function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, $precision) . ' ' . $units[$i];
    }
}

