<?php

namespace App\Http\Controllers;

use App\Models\Petani;
use App\Models\Desa;
use App\Models\Poktan;
use App\Models\Kios;
use App\Services\PetaniService;
use App\Services\ActivityLogger;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\PetaniExport;
use App\Imports\PetaniImport;

class PetaniController extends Controller
{
    /**
     * @var PetaniService
     */
    protected $petaniService;
    
    /**
     * PetaniController constructor.
     * 
     * @param PetaniService $petaniService
     */
    public function __construct(PetaniService $petaniService)
    {
        $this->petaniService = $petaniService;
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $petanis = $this->petaniService->getAllPetani();
        return view('petani.index', compact('petanis'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $desas = Desa::all();
        $poktans = Poktan::all();
        $kios = Kios::all();
        return view('petani.create', compact('desas', 'poktans', 'kios'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Gunakan service untuk membuat data petani baru
            $petani = $this->petaniService->createPetani($request->all());
            
            return redirect()->route('petani.index')
                ->with('success', 'Data petani berhasil ditambahkan.');
        } catch (ValidationException $e) {
            return redirect()->route('petani.create')
                ->withErrors($e->validator)
                ->withInput();
        } catch (\Exception $e) {
            // Tangani error data truncated
            if (strpos($e->getMessage(), 'Data truncated') !== false) {
                return redirect()->back()
                    ->with('error', 'Terjadi kesalahan: Data tidak sesuai dengan format yang diharapkan.')
                    ->withInput();
            }
            
            // Tangani error duplikat
            if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                return redirect()->route('petani.create')
                    ->with('error', 'Nomor KTP sudah terdaftar dalam sistem.')
                    ->withInput();
            }
            
            return redirect()->route('petani.create')
                ->with('error', 'Terjadi kesalahan saat menyimpan data: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $petani = $this->petaniService->getPetaniById($id);
            
            if (!$petani) {
                return redirect()->route('petani.index')
                    ->with('error', 'Data petani tidak ditemukan.');
            }
            
            return view('petani.show', compact('petani'));
        } catch (\Exception $e) {
            return redirect()->route('petani.index')
                ->with('error', 'Terjadi kesalahan saat mengakses data petani: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        try {
            $petani = $this->petaniService->getPetaniById($id);
            
            if (!$petani) {
                return redirect()->route('petani.index')
                    ->with('error', 'Data petani tidak ditemukan.');
            }
            
            $desas = Desa::all();
            $poktans = Poktan::all();
            $kios = Kios::all();
            return view('petani.edit', compact('petani', 'desas', 'poktans', 'kios'));
        } catch (\Exception $e) {
            return redirect()->route('petani.index')
                ->with('error', 'Terjadi kesalahan saat mengakses data petani: ' . $e->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            // Gunakan service untuk mengupdate data petani
            $petani = $this->petaniService->updatePetani($id, $request->all());
            
            return redirect()->route('petani.index')
                ->with('success', 'Data petani berhasil diperbarui.');
        } catch (ValidationException $e) {
            return redirect()->route('petani.edit', $id)
                ->withErrors($e->validator)
                ->withInput();
        } catch (\Exception $e) {
            // Tangani error data truncated
            if (strpos($e->getMessage(), 'Data truncated') !== false) {
                return redirect()->route('petani.edit', $id)
                    ->with('error', 'Terjadi kesalahan: Data tidak sesuai dengan format yang diharapkan.')
                    ->withInput();
            }
            
            // Tangani error duplikat
            if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                return redirect()->route('petani.edit', $id)
                    ->with('error', 'Nomor KTP sudah terdaftar dalam sistem.')
                    ->withInput();
            }
            
            // Tangani error data tidak ditemukan
            if (strpos($e->getMessage(), 'tidak ditemukan') !== false) {
                return redirect()->route('petani.index')
                    ->with('error', $e->getMessage());
            }
            
            return redirect()->route('petani.edit', $id)
                ->with('error', 'Terjadi kesalahan saat memperbarui data petani: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $this->petaniService->deletePetani($id);
            return redirect()->route('petani.index')
                ->with('success', 'Data petani berhasil dihapus.');
        } catch (\Exception $e) {
            return redirect()->route('petani.index')
                ->with('error', 'Terjadi kesalahan saat menghapus data petani: ' . $e->getMessage());
        }
    }

    /**
     * Export petani data to Excel.
     */
    public function export() 
    {
        // Log export activity untuk audit
        $recordCount = Petani::count();
        ActivityLogger::logExport('petani', $recordCount);
        
        return Excel::download(new PetaniExport, 'petani.xlsx');
    }

    /**
     * Import petani data from Excel.
     */
    public function import(Request $request) 
    {
        // SECURITY: Enhanced file upload validation
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls|max:2048', // Max 2MB
        ], [
            'file.required' => 'File import wajib diupload.',
            'file.mimes' => 'File harus berformat Excel (xlsx atau xls).',
            'file.max' => 'Ukuran file maksimal 2MB.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $beforeCount = Petani::count();
            Excel::import(new PetaniImport, $request->file('file'));
            $afterCount = Petani::count();
            $imported = $afterCount - $beforeCount;
            
            // Log import activity untuk audit
            ActivityLogger::logImport('petani', $imported, true);
            
            return redirect()->route('petani.index')
                ->with('success', 'Data petani berhasil diimpor.');
        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errorMessages = [];
            
            foreach ($failures as $failure) {
                $row = $failure->row();
                $attribute = $failure->attribute();
                $errors = $failure->errors();
                
                foreach ($errors as $error) {
                    $errorMessages[] = "Baris {$row}: {$attribute} - {$error}";
                }
            }
            
            // Log failed import
            ActivityLogger::logImport('petani', 0, false);
            
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat mengimpor data: <br>' . implode('<br>', $errorMessages))
                ->withInput();
        } catch (\Exception $e) {
            // Log failed import
            ActivityLogger::logImport('petani', 0, false);
            // Tangani error data truncated
            if (strpos($e->getMessage(), 'Data truncated') !== false) {
                return redirect()->back()
                    ->with('error', 'Terjadi kesalahan: Data tidak sesuai dengan format yang diharapkan. Pastikan status adalah "terdaftar" atau "tidak_terdaftar".')
                    ->withInput();
            }
            
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat mengimpor data: ' . $e->getMessage())
                ->withInput();
        }
    }
}
