<?php

namespace App\Http\Controllers;

use App\Models\Poktan;
use App\Models\Desa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\PoktanExport;
use App\Imports\PoktanImport;
use Illuminate\Http\JsonResponse;

class PoktanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $poktans = Poktan::with('desa')->paginate(10);
        return view('poktan.index', compact('poktans'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $desas = Desa::all();
        return view('poktan.create', compact('desas'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id_poktan' => 'required|string|max:20|unique:poktans,id_poktan',
            'nama_poktan' => 'required|string|max:100',
            'kode_desa' => 'required|string|exists:desas,kode_desa',
        ]);

        if ($validator->fails()) {
            return redirect()->route('poktan.create')
                ->withErrors($validator)
                ->withInput();
        }

        Poktan::create($request->all());

        return redirect()->route('poktan.index')
            ->with('success', 'Kelompok Tani berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $poktan = Poktan::with('desa')->findOrFail($id);
        return view('poktan.show', compact('poktan'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $poktan = Poktan::findOrFail($id);
        $desas = Desa::all();
        return view('poktan.edit', compact('poktan', 'desas'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $poktan = Poktan::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'id_poktan' => ['required', 'string', 'max:20', Rule::unique('poktans')->ignore($id, 'id_poktan')],
            'nama_poktan' => 'required|string|max:100',
            'kode_desa' => 'required|string|exists:desas,kode_desa',
        ]);

        if ($validator->fails()) {
            return redirect()->route('poktan.edit', $id)
                ->withErrors($validator)
                ->withInput();
        }

        $poktan->update($request->all());

        return redirect()->route('poktan.index')
            ->with('success', 'Kelompok Tani berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $poktan = Poktan::findOrFail($id);
        $poktan->delete();

        return redirect()->route('poktan.index')
            ->with('success', 'Kelompok Tani berhasil dihapus.');
    }

    /**
     * Export data kelompok tani ke Excel
     */
    public function export()
    {
        return Excel::download(new PoktanExport(), 'daftar-kelompok-tani-' . date('Y-m-d') . '.xlsx');
    }

    /**
     * Import data kelompok tani dari Excel
     */
    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xlsx,xls|max:5120', // Max 5MB
        ]);

        if ($validator->fails()) {
            return redirect()->route('poktan.index')
                ->withErrors($validator)
                ->with('error', 'File tidak valid. Pastikan file berformat Excel (.xlsx atau .xls) dan maksimal 5MB.');
        }

        try {
            $import = new PoktanImport();
            Excel::import($import, $request->file('file'));

            $failures = $import->failures();
            $errors = $import->errors();

            if (count($failures) > 0 || count($errors) > 0) {
                $errorMessages = [];
                
                foreach ($failures as $failure) {
                    $errorMessages[] = "Baris {$failure->row()}: " . implode(', ', $failure->errors());
                }
                
                foreach ($errors as $error) {
                    $errorMessages[] = $error->getMessage();
                }
                
                return redirect()->route('poktan.index')
                    ->with('warning', 'Import selesai dengan beberapa kesalahan: ' . implode(' | ', $errorMessages));
            }

            return redirect()->route('poktan.index')
                ->with('success', 'Data Kelompok Tani berhasil diimport!');
                
        } catch (\Exception $e) {
            return redirect()->route('poktan.index')
                ->with('error', 'Terjadi kesalahan saat import: ' . $e->getMessage());
        }
    }

    /**
     * Download template Excel untuk import kelompok tani
     */
    public function downloadTemplate()
    {
        return Excel::download(new PoktanExport(true), 'template-kelompok-tani.xlsx');
    }

    /**
     * Mengembalikan daftar poktan dalam bentuk JSON berdasarkan kode desa
     */
    public function byDesa(string $kode_desa): JsonResponse
    {
        $poktans = Poktan::where('kode_desa', $kode_desa)
            ->orderBy('nama_poktan')
            ->get(['id_poktan', 'nama_poktan']);

        return response()->json($poktans);
    }
}
