<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use App\Rules\StrongPassword;
use App\Services\ActivityLogger;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $users = User::orderBy('created_at', 'desc')->paginate(10);
        return view('user.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('user.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', new StrongPassword()],
            'role' => ['required', 'in:admin,user'],
            'unit_kerja' => ['nullable', 'string', 'max:255'],
        ]);

        User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
            'unit_kerja' => $validated['unit_kerja'],
        ]);

        return redirect()->route('user.index')
            ->with('success', 'User berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return view('user.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        return view('user.edit', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'role' => ['required', 'in:admin,user'],
            'unit_kerja' => ['nullable', 'string', 'max:255'],
            'password' => ['nullable', 'confirmed', new StrongPassword()],
        ]);

        // Track role changes untuk security audit
        $oldRole = $user->role;
        $roleChanged = ($oldRole !== $validated['role']);
        
        $user->name = $validated['name'];
        $user->email = $validated['email'];
        $user->role = $validated['role'];
        $user->unit_kerja = $validated['unit_kerja'];

        $passwordChanged = false;
        if ($request->filled('password')) {
            $user->password = Hash::make($validated['password']);
            $passwordChanged = true;
        }

        $user->save();
        
        // Log security events
        if ($roleChanged) {
            ActivityLogger::logRoleChange($user->id, $oldRole, $validated['role']);
        }
        
        if ($passwordChanged && $user->id !== auth()->id()) {
            ActivityLogger::logPasswordChange($user->id);
        }

        return redirect()->route('user.index')
            ->with('success', 'User berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Prevent deleting own account
        if ($user->id === auth()->id()) {
            return redirect()->route('user.index')
                ->with('error', 'Anda tidak dapat menghapus akun Anda sendiri.');
        }

        // Store info untuk logging sebelum delete
        $userId = $user->id;
        $userEmail = $user->email;
        
        $user->delete();
        
        // Log security event
        ActivityLogger::logUserDeletion($userId, $userEmail);

        return redirect()->route('user.index')
            ->with('success', 'User berhasil dihapus.');
    }
}

