<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Middleware SecurityHeaders
 * 
 * Menambahkan security headers untuk melindungi aplikasi dari berbagai
 * serangan web seperti XSS, clickjacking, MIME sniffing, dll.
 * Sesuai dengan best practices OWASP dan standar keamanan modern
 */
class SecurityHeaders
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // X-Frame-Options: Mencegah clickjacking attack
        // DENY = tidak boleh di-frame sama sekali
        $response->headers->set('X-Frame-Options', 'DENY');

        // X-Content-Type-Options: Mencegah MIME sniffing
        // nosniff = browser tidak boleh menebak content type
        $response->headers->set('X-Content-Type-Options', 'nosniff');

        // X-XSS-Protection: Aktifkan XSS filter browser (untuk browser lama)
        // mode=block = block page jika XSS terdeteksi
        $response->headers->set('X-XSS-Protection', '1; mode=block');

        // Referrer-Policy: Kontrol informasi referrer yang dikirim
        // strict-origin-when-cross-origin = kirim origin saja untuk cross-origin
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');

        // Permissions-Policy: Kontrol fitur browser yang dapat diakses
        // Disable fitur yang tidak diperlukan
        $response->headers->set('Permissions-Policy', 
            'geolocation=(), microphone=(), camera=(), payment=(), usb=(), magnetometer=(), gyroscope=(), accelerometer=()'
        );

        // Content-Security-Policy (CSP): Proteksi komprehensif terhadap XSS dan injection attacks
        $csp = $this->getContentSecurityPolicy();
        $response->headers->set('Content-Security-Policy', $csp);

        // Strict-Transport-Security (HSTS): Force HTTPS untuk semua koneksi
        // max-age=31536000 = 1 tahun
        // includeSubDomains = berlaku untuk semua subdomain
        // preload = masukkan ke HSTS preload list browser
        if ($request->secure() || app()->environment('production')) {
            $response->headers->set('Strict-Transport-Security', 
                'max-age=31536000; includeSubDomains; preload'
            );
        }

        // Remove server information disclosure
        $response->headers->remove('X-Powered-By');
        $response->headers->remove('Server');

        return $response;
    }

    /**
     * Generate Content Security Policy header
     * 
     * @return string
     */
    protected function getContentSecurityPolicy(): string
    {
        // CSP directives untuk proteksi maksimal
        $directives = [
            // Default: hanya dari origin yang sama
            "default-src 'self'",
            
            // Script: hanya dari self, dengan inline script untuk Laravel/Alpine.js
            // unsafe-inline dan unsafe-eval diperlukan untuk beberapa library
            // Di production, sebaiknya gunakan nonce atau hash untuk inline script
            "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.jsdelivr.net",
            
            // Style: allow inline styles untuk framework CSS
            "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com",
            
            // Fonts: dari self dan Google Fonts
            "font-src 'self' https://fonts.gstatic.com data:",
            
            // Images: dari self dan data URIs
            "img-src 'self' data: https:",
            
            // AJAX/Fetch: hanya ke origin sendiri
            "connect-src 'self'",
            
            // Frames: tidak boleh ada frame (sama dengan X-Frame-Options)
            "frame-ancestors 'none'",
            
            // Forms: hanya submit ke origin sendiri
            "form-action 'self'",
            
            // Base tag: tidak boleh diubah
            "base-uri 'self'",
            
            // Object/Embed: tidak diizinkan (Flash, Java applet, dll)
            "object-src 'none'",
        ];

        return implode('; ', $directives);
    }
}

