<?php

namespace App\Repositories;

use App\Models\Petani;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;

/**
 * Class PetaniRepository
 * 
 * Implementasi repository untuk operasi CRUD petani
 */
class PetaniRepository implements PetaniRepositoryInterface
{
    /**
     * @var Petani
     */
    protected $model;
    
    /**
     * PetaniRepository constructor.
     * 
     * @param Petani $model
     */
    public function __construct(Petani $model)
    {
        $this->model = $model;
    }
    
    /**
     * {@inheritdoc}
     */
    public function getAll(): Collection
    {
        return $this->model->with(['desa', 'poktan', 'kios'])->get();
    }
    
    /**
     * {@inheritdoc}
     */
    public function findById(string $id): ?Petani
    {
        return $this->model->with(['desa', 'poktan', 'kios'])->find($id);
    }
    
    /**
     * {@inheritdoc}
     */
    public function findByNoKtp(string $noKtp): ?Petani
    {
        return $this->model->where('no_ktp', $noKtp)->first();
    }
    
    /**
     * {@inheritdoc}
     */
    public function create(array $data): Petani
    {
        return $this->model->create($data);
    }
    
    /**
     * {@inheritdoc}
     */
    public function update(string $id, array $data): Petani
    {
        $petani = $this->findById($id);
        
        if (!$petani) {
            throw new \Exception("Petani dengan ID {$id} tidak ditemukan");
        }
        
        $petani->update($data);
        return $petani->fresh();
    }
    
    /**
     * {@inheritdoc}
     */
    public function delete(string $id): bool
    {
        $petani = $this->findById($id);
        
        if (!$petani) {
            throw new \Exception("Petani dengan ID {$id} tidak ditemukan");
        }
        
        return $petani->delete();
    }
    
    /**
     * {@inheritdoc}
     */
    public function getByDesa(string $kodeDesa): Collection
    {
        return $this->model->where('kode_desa', $kodeDesa)
            ->with(['desa', 'poktan', 'kios'])
            ->get();
    }
    
    /**
     * {@inheritdoc}
     */
    public function getByPoktan(string $idPoktan): Collection
    {
        return $this->model->where('id_poktan', $idPoktan)
            ->with(['desa', 'poktan', 'kios'])
            ->get();
    }
    
    /**
     * {@inheritdoc}
     */
    public function getByKios(string $kodeKios): Collection
    {
        return $this->model->where('kode_kios', $kodeKios)
            ->with(['desa', 'poktan', 'kios'])
            ->get();
    }
}