<?php

namespace App\Rules;

use Closure;
use Illuminate\Contracts\Validation\ValidationRule;

/**
 * Rule StrongPassword
 * 
 * Custom validation rule untuk password yang kuat
 * Memenuhi standar keamanan tinggi:
 * - Minimal 12 karakter
 * - Minimal 1 huruf besar (uppercase)
 * - Minimal 1 huruf kecil (lowercase)
 * - Minimal 1 angka (number)
 * - Minimal 1 simbol/special character
 */
class StrongPassword implements ValidationRule
{
    /**
     * Run the validation rule.
     *
     * @param  \Closure(string): \Illuminate\Translation\PotentiallyTranslatedString  $fail
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        // Cek panjang minimal 12 karakter
        if (strlen($value) < 12) {
            $fail('Password harus minimal 12 karakter.');
            return;
        }

        // Cek apakah ada huruf besar
        if (!preg_match('/[A-Z]/', $value)) {
            $fail('Password harus mengandung minimal 1 huruf besar (A-Z).');
            return;
        }

        // Cek apakah ada huruf kecil
        if (!preg_match('/[a-z]/', $value)) {
            $fail('Password harus mengandung minimal 1 huruf kecil (a-z).');
            return;
        }

        // Cek apakah ada angka
        if (!preg_match('/[0-9]/', $value)) {
            $fail('Password harus mengandung minimal 1 angka (0-9).');
            return;
        }

        // Cek apakah ada simbol/special character
        if (!preg_match('/[^A-Za-z0-9]/', $value)) {
            $fail('Password harus mengandung minimal 1 simbol (!@#$%^&*, dll).');
            return;
        }

        // Cek password yang terlalu umum/mudah ditebak
        $commonPasswords = [
            'Password123!',
            'Admin123456!',
            'Welcome123!',
            'Qwerty123456!',
            '123456789012!',
        ];

        if (in_array($value, $commonPasswords)) {
            $fail('Password yang Anda pilih terlalu umum. Silakan pilih password yang lebih unik.');
            return;
        }
    }
}

