<?php

namespace App\Services;

use App\Models\Petani;
use App\Repositories\PetaniRepositoryInterface;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Validation\Rule;

/**
 * Class PetaniService
 * 
 * Service layer untuk menangani logika bisnis terkait petani
 */
class PetaniService
{
    /**
     * @var PetaniRepositoryInterface
     */
    protected $petaniRepository;
    
    /**
     * PetaniService constructor.
     * 
     * @param PetaniRepositoryInterface $petaniRepository
     */
    public function __construct(PetaniRepositoryInterface $petaniRepository)
    {
        $this->petaniRepository = $petaniRepository;
    }
    
    /**
     * Mendapatkan semua data petani
     * 
     * @return Collection
     */
    public function getAllPetani(): Collection
    {
        return $this->petaniRepository->getAll();
    }
    
    /**
     * Mendapatkan data petani berdasarkan ID
     * 
     * @param string $id
     * @return Petani|null
     */
    public function getPetaniById(string $id): ?Petani
    {
        return $this->petaniRepository->findById($id);
    }
    
    /**
     * Membuat data petani baru dengan validasi
     * 
     * @param array $data
     * @return Petani
     * @throws ValidationException
     */
    public function createPetani(array $data): Petani
    {
        // Validasi data
        $validator = Validator::make($data, [
            'nama_penyuluh' => 'required|string|max:100',
            'no_ktp' => 'required|string|size:16|unique:petanis,no_ktp',
            'nama_ibu_kandung' => 'required|string|max:100',
            'kode_desa' => 'required|string|exists:desas,kode_desa',
            'id_poktan' => [
                'required','string',
                Rule::exists('poktans','id_poktan')->where(function($q) use ($data) {
                    return $q->where('kode_desa', $data['kode_desa'] ?? null);
                }),
            ],
            'kode_kios' => 'required|string|exists:kios,kode_kios',
            'subsektor' => 'nullable|in:tanaman_pangan,hortikultura,perkebunan,peternakan',
            'komoditas' => 'nullable|string|max:100',
            'luas_lahan' => 'nullable|numeric|min:0.01',
            'status' => 'nullable|in:belum_terinput,sudah_terinput',
            'alamat' => 'nullable|string|max:255',
            'no_hp' => 'nullable|string|max:15',
            'email' => 'nullable|email|max:100',
            'jenis_kelamin' => 'nullable|in:L,P',
            'tanggal_lahir' => 'nullable|date',
        ]);
        
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
        
        // Catat waktu mulai untuk logging
        $startTime = now();
        
        try {
            // Gunakan transaction untuk memastikan integritas data
            return DB::transaction(function () use ($data, $startTime) {
                // Siapkan data yang akan disimpan
                $petaniData = $data;
                
                // Tambahkan nilai default untuk field opsional
                if (empty($data['subsektor'])) {
                    $petaniData['subsektor'] = null;
                }
                if (empty($data['komoditas'])) {
                    $petaniData['komoditas'] = null;
                }
                if (empty($data['luas_lahan'])) {
                    $petaniData['luas_lahan'] = 0.01;
                }
                if (empty($data['status'])) {
                    $petaniData['status'] = 'belum_terinput';
                }
                
                // Buat data petani baru
                $petani = $this->petaniRepository->create($petaniData);
                
                // Log sukses (tanpa data sensitif untuk keamanan)
                Log::info('Data petani berhasil disimpan', [
                    'id' => $petani->id,
                    'nama' => $petani->nama_penyuluh,
                    'timestamp' => now()->format('Y-m-d H:i:s'),
                    'user_id' => auth()->id(),
                    'execution_time' => now()->diffInMilliseconds($startTime) . 'ms'
                ]);
                
                return $petani;
            });
        } catch (\Exception $e) {
            // Log error
            Log::error('Gagal menyimpan data petani', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'input' => $data,
                'timestamp' => now()->format('Y-m-d H:i:s'),
                'user_id' => auth()->id()
            ]);
            
            throw $e;
        }
    }
    
    /**
     * Mengupdate data petani dengan validasi
     * 
     * @param string $id
     * @param array $data
     * @return Petani
     * @throws ValidationException
     */
    public function updatePetani(string $id, array $data): Petani
    {
        // Validasi data
        $validator = Validator::make($data, [
            'nama_penyuluh' => 'required|string|max:100',
            'no_ktp' => 'required|string|size:16|unique:petanis,no_ktp,'.$id,
            'nama_ibu_kandung' => 'required|string|max:100',
            'kode_desa' => 'required|string|exists:desas,kode_desa',
            'id_poktan' => [
                'required','string',
                Rule::exists('poktans','id_poktan')->where(function($q) use ($data) {
                    return $q->where('kode_desa', $data['kode_desa'] ?? null);
                }),
            ],
            'kode_kios' => 'required|string|exists:kios,kode_kios',
            'subsektor' => 'nullable|in:tanaman_pangan,hortikultura,perkebunan,peternakan',
            'komoditas' => 'nullable|string|max:100',
            'luas_lahan' => 'nullable|numeric|min:0.01',
            'status' => 'required|in:belum_terinput,sudah_terinput',
            'alamat' => 'nullable|string|max:255',
            'no_hp' => 'nullable|string|max:15',
            'email' => 'nullable|email|max:100',
            'jenis_kelamin' => 'nullable|in:L,P',
            'tanggal_lahir' => 'nullable|date',
        ]);
        
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
        
        // Catat waktu mulai untuk logging
        $startTime = now();
        
        try {
            // Gunakan transaction untuk memastikan integritas data
            return DB::transaction(function () use ($id, $data, $startTime) {
                // Siapkan data untuk update
                $updateData = $data;
                
                // Handle nullable fields
                if (empty($data['subsektor'])) {
                    $updateData['subsektor'] = null;
                }
                if (empty($data['komoditas'])) {
                    $updateData['komoditas'] = null;
                }
                if (empty($data['luas_lahan'])) {
                    $updateData['luas_lahan'] = 0.01;
                }
                
                // Update data petani
                $petani = $this->petaniRepository->update($id, $updateData);
                
                // Log sukses (tanpa data sensitif untuk keamanan)
                Log::info('Data petani berhasil diperbarui', [
                    'id' => $petani->id,
                    'nama' => $petani->nama_penyuluh,
                    'timestamp' => now()->format('Y-m-d H:i:s'),
                    'user_id' => auth()->id(),
                    'execution_time' => now()->diffInMilliseconds($startTime) . 'ms'
                ]);
                
                return $petani;
            });
        } catch (\Exception $e) {
            // Log error
            Log::error('Gagal memperbarui data petani', [
                'id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'input' => $data,
                'timestamp' => now()->format('Y-m-d H:i:s'),
                'user_id' => auth()->id()
            ]);
            
            throw $e;
        }
    }
    
    /**
     * Menghapus data petani
     * 
     * @param string $id
     * @return bool
     */
    public function deletePetani(string $id): bool
    {
        try {
            return DB::transaction(function () use ($id) {
                return $this->petaniRepository->delete($id);
            });
        } catch (\Exception $e) {
            Log::error('Gagal menghapus data petani', [
                'id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'timestamp' => now()->format('Y-m-d H:i:s'),
                'user_id' => auth()->id()
            ]);
            
            throw $e;
        }
    }
    
    /**
     * Mendapatkan data petani berdasarkan desa
     * 
     * @param string $kodeDesa
     * @return Collection
     */
    public function getPetaniByDesa(string $kodeDesa): Collection
    {
        return $this->petaniRepository->getByDesa($kodeDesa);
    }
    
    /**
     * Mendapatkan data petani berdasarkan kelompok tani
     * 
     * @param string $idPoktan
     * @return Collection
     */
    public function getPetaniByPoktan(string $idPoktan): Collection
    {
        return $this->petaniRepository->getByPoktan($idPoktan);
    }
    
    /**
     * Mendapatkan data petani berdasarkan kios
     * 
     * @param string $kodeKios
     * @return Collection
     */
    public function getPetaniByKios(string $kodeKios): Collection
    {
        return $this->petaniRepository->getByKios($kodeKios);
    }
}