<?php

namespace App\Traits;

use Illuminate\Support\Facades\Crypt;
use Illuminate\Contracts\Encryption\DecryptException;

/**
 * Trait EncryptsAttributes
 * 
 * Trait untuk enkripsi otomatis field sensitif di database
 * Menggunakan Laravel's Crypt facade untuk AES-256 encryption
 */
trait EncryptsAttributes
{
    /**
     * Boot the trait
     */
    public static function bootEncryptsAttributes()
    {
        // Enkripsi sebelum menyimpan ke database
        static::creating(function ($model) {
            $model->encryptAttributes();
        });

        static::updating(function ($model) {
            $model->encryptAttributes();
        });

        // Dekripsi setelah mengambil dari database
        static::retrieved(function ($model) {
            $model->decryptAttributes();
        });
    }

    /**
     * Enkripsi atribut yang ditentukan
     */
    protected function encryptAttributes()
    {
        foreach ($this->getEncryptableAttributes() as $attribute) {
            if (isset($this->attributes[$attribute]) && !empty($this->attributes[$attribute])) {
                // Cek apakah sudah terenkripsi (untuk update)
                if (!$this->isEncrypted($this->attributes[$attribute])) {
                    $this->attributes[$attribute] = Crypt::encryptString($this->attributes[$attribute]);
                }
            }
        }
    }

    /**
     * Dekripsi atribut yang ditentukan
     */
    protected function decryptAttributes()
    {
        foreach ($this->getEncryptableAttributes() as $attribute) {
            if (isset($this->attributes[$attribute]) && !empty($this->attributes[$attribute])) {
                try {
                    // Cek apakah terenkripsi sebelum dekripsi
                    if ($this->isEncrypted($this->attributes[$attribute])) {
                        $this->attributes[$attribute] = Crypt::decryptString($this->attributes[$attribute]);
                    }
                } catch (DecryptException $e) {
                    // Jika dekripsi gagal, log error tapi tetap return value asli
                    \Log::error('Decryption failed for attribute: ' . $attribute, [
                        'model' => get_class($this),
                        'id' => $this->id ?? 'unknown',
                        'error' => $e->getMessage()
                    ]);
                }
            }
        }
    }

    /**
     * Cek apakah string sudah terenkripsi
     * 
     * @param string $value
     * @return bool
     */
    protected function isEncrypted($value)
    {
        // Laravel encrypted string dimulai dengan base64 encoded data
        // dan mengandung payload JSON dengan iv, value, mac
        if (empty($value)) {
            return false;
        }

        try {
            $payload = json_decode(base64_decode($value), true);
            return is_array($payload) && 
                   isset($payload['iv']) && 
                   isset($payload['value']) && 
                   isset($payload['mac']);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Get atribut yang harus dienkripsi
     * 
     * @return array
     */
    protected function getEncryptableAttributes()
    {
        return $this->encryptable ?? [];
    }

    /**
     * Accessor untuk atribut terenkripsi
     * Digunakan ketika attribute diakses tapi belum didekripsi
     */
    public function getAttribute($key)
    {
        $value = parent::getAttribute($key);

        // Jika attribute ada di daftar encryptable dan value terenkripsi, dekripsi
        if (in_array($key, $this->getEncryptableAttributes()) && 
            !empty($value) && 
            $this->isEncrypted($value)) {
            try {
                return Crypt::decryptString($value);
            } catch (DecryptException $e) {
                \Log::error('Decryption failed on getAttribute: ' . $key, [
                    'model' => get_class($this),
                    'id' => $this->id ?? 'unknown'
                ]);
                return null;
            }
        }

        return $value;
    }

    /**
     * Override toArray untuk memastikan data terenkripsi di-handle dengan benar
     */
    public function toArray()
    {
        $array = parent::toArray();

        // Pastikan encryptable attributes di-mask untuk keamanan tambahan
        // jika diperlukan di masa depan
        return $array;
    }

    /**
     * Disable enkripsi sementara untuk operasi batch
     * Hati-hati menggunakan method ini
     */
    public function withoutEncryption(callable $callback)
    {
        $original = $this->encryptable;
        $this->encryptable = [];
        
        $result = $callback($this);
        
        $this->encryptable = $original;
        
        return $result;
    }
}

