<?php

namespace Tests\Feature;

use App\Models\User;
use App\Services\PetaniService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Validation\ValidationException;
use Mockery;
use Tests\TestCase;

class PetaniControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $petaniService;

    public function setUp(): void
    {
        parent::setUp();
        $this->petaniService = Mockery::mock(PetaniService::class);
        $this->app->instance(PetaniService::class, $this->petaniService);
    }

    public function tearDown(): void
    {
        Mockery::close();
        parent::tearDown();
    }

    /** @test */
    public function store_method_creates_petani_and_redirects_with_success_message()
    {
        // Arrange
        $user = User::factory()->create();
        $this->actingAs($user);

        $petaniData = [
            'nama_penyuluh' => 'Test Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 1.5,
            'status' => 'belum_terinput'
        ];

        // Mock service
        $this->petaniService->shouldReceive('createPetani')
            ->once()
            ->with(Mockery::on(function ($data) use ($petaniData) {
                return $data['nama_penyuluh'] === $petaniData['nama_penyuluh'] &&
                       $data['no_ktp'] === $petaniData['no_ktp'];
            }))
            ->andReturn((object) array_merge(['id_petani' => 'TEST001'], $petaniData));

        // Act
        $response = $this->post(route('petani.store'), $petaniData);

        // Assert
        $response->assertRedirect(route('petani.index'));
        $response->assertSessionHas('success', 'Data petani berhasil disimpan.');
    }

    /** @test */
    public function store_method_redirects_back_with_errors_on_validation_exception()
    {
        // Arrange
        $user = User::factory()->create();
        $this->actingAs($user);

        $petaniData = [
            'nama_penyuluh' => 'Test Petani',
            'no_ktp' => '123456', // Invalid: too short
        ];

        // Mock service to throw ValidationException
        $this->petaniService->shouldReceive('createPetani')
            ->once()
            ->andThrow(ValidationException::withMessages([
                'no_ktp' => ['Nomor KTP harus 16 digit']
            ]));

        // Act
        $response = $this->post(route('petani.store'), $petaniData);

        // Assert
        $response->assertRedirect();
        $response->assertSessionHasErrors(['no_ktp']);
    }

    /** @test */
    public function update_method_updates_petani_and_redirects_with_success_message()
    {
        // Arrange
        $user = User::factory()->create();
        $this->actingAs($user);

        $petaniId = 'TEST001';
        $petaniData = [
            'nama_penyuluh' => 'Updated Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 2.0,
            'status' => 'sudah_terinput'
        ];

        // Mock service
        $this->petaniService->shouldReceive('updatePetani')
            ->once()
            ->with($petaniId, Mockery::on(function ($data) use ($petaniData) {
                return $data['nama_penyuluh'] === $petaniData['nama_penyuluh'] &&
                       $data['luas_lahan'] === $petaniData['luas_lahan'];
            }))
            ->andReturn((object) array_merge(['id_petani' => $petaniId], $petaniData));

        // Act
        $response = $this->put(route('petani.update', $petaniId), $petaniData);

        // Assert
        $response->assertRedirect(route('petani.index'));
        $response->assertSessionHas('success', 'Data petani berhasil diperbarui.');
    }

    /** @test */
    public function destroy_method_deletes_petani_and_redirects_with_success_message()
    {
        // Arrange
        $user = User::factory()->create();
        $this->actingAs($user);

        $petaniId = 'TEST001';

        // Mock service
        $this->petaniService->shouldReceive('deletePetani')
            ->once()
            ->with($petaniId)
            ->andReturn(true);

        // Act
        $response = $this->delete(route('petani.destroy', $petaniId));

        // Assert
        $response->assertRedirect(route('petani.index'));
        $response->assertSessionHas('success', 'Data petani berhasil dihapus.');
    }

    /** @test */
    public function destroy_method_redirects_with_error_message_on_exception()
    {
        // Arrange
        $user = User::factory()->create();
        $this->actingAs($user);

        $petaniId = 'NONEXISTENT';

        // Mock service to throw exception
        $this->petaniService->shouldReceive('deletePetani')
            ->once()
            ->with($petaniId)
            ->andThrow(new \Exception('Petani dengan ID NONEXISTENT tidak ditemukan'));

        // Act
        $response = $this->delete(route('petani.destroy', $petaniId));

        // Assert
        $response->assertRedirect(route('petani.index'));
        $response->assertSessionHas('error', 'Terjadi kesalahan saat menghapus data petani: Petani dengan ID NONEXISTENT tidak ditemukan');
    }
}