<?php

namespace Tests\Unit;

use App\Models\Petani;
use App\Repositories\PetaniRepository;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class PetaniRepositoryTest extends TestCase
{
    use RefreshDatabase;

    protected $repository;

    public function setUp(): void
    {
        parent::setUp();
        $this->repository = new PetaniRepository();
    }

    /** @test */
    public function it_can_create_petani()
    {
        // Arrange
        $data = [
            'nama_penyuluh' => 'Test Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 1.5,
            'status' => 'belum_terinput'
        ];

        // Act
        $petani = $this->repository->create($data);

        // Assert
        $this->assertInstanceOf(Petani::class, $petani);
        $this->assertEquals('Test Petani', $petani->nama_penyuluh);
        $this->assertEquals('1234567890123456', $petani->no_ktp);
        $this->assertEquals('tanaman_pangan', $petani->subsektor);
        $this->assertEquals(1.5, $petani->luas_lahan);
    }

    /** @test */
    public function it_can_find_petani_by_id()
    {
        // Arrange
        $petani = Petani::create([
            'nama_penyuluh' => 'Test Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 1.5,
            'status' => 'belum_terinput'
        ]);

        // Act
        $foundPetani = $this->repository->findById($petani->id_petani);

        // Assert
        $this->assertInstanceOf(Petani::class, $foundPetani);
        $this->assertEquals($petani->id_petani, $foundPetani->id_petani);
        $this->assertEquals('Test Petani', $foundPetani->nama_penyuluh);
    }

    /** @test */
    public function it_returns_null_when_petani_not_found()
    {
        // Act
        $result = $this->repository->findById('nonexistent-id');

        // Assert
        $this->assertNull($result);
    }

    /** @test */
    public function it_can_update_petani()
    {
        // Arrange
        $petani = Petani::create([
            'nama_penyuluh' => 'Test Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 1.5,
            'status' => 'belum_terinput'
        ]);

        $updateData = [
            'nama_penyuluh' => 'Updated Petani',
            'luas_lahan' => 2.0,
            'status' => 'sudah_terinput'
        ];

        // Act
        $updatedPetani = $this->repository->update($petani->id_petani, $updateData);

        // Assert
        $this->assertInstanceOf(Petani::class, $updatedPetani);
        $this->assertEquals($petani->id_petani, $updatedPetani->id_petani);
        $this->assertEquals('Updated Petani', $updatedPetani->nama_penyuluh);
        $this->assertEquals(2.0, $updatedPetani->luas_lahan);
        $this->assertEquals('sudah_terinput', $updatedPetani->status);
        $this->assertEquals('1234567890123456', $updatedPetani->no_ktp); // Unchanged field
    }

    /** @test */
    public function it_throws_exception_when_updating_nonexistent_petani()
    {
        // Arrange
        $updateData = [
            'nama_penyuluh' => 'Updated Petani',
        ];

        // Act & Assert
        $this->expectException(\Exception::class);
        $this->repository->update('nonexistent-id', $updateData);
    }

    /** @test */
    public function it_can_delete_petani()
    {
        // Arrange
        $petani = Petani::create([
            'nama_penyuluh' => 'Test Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 1.5,
            'status' => 'belum_terinput'
        ]);

        // Act
        $result = $this->repository->delete($petani->id_petani);

        // Assert
        $this->assertTrue($result);
        $this->assertNull($this->repository->findById($petani->id_petani));
    }

    /** @test */
    public function it_throws_exception_when_deleting_nonexistent_petani()
    {
        // Act & Assert
        $this->expectException(\Exception::class);
        $this->repository->delete('nonexistent-id');
    }

    /** @test */
    public function it_can_get_all_petani()
    {
        // Arrange
        Petani::create([
            'nama_penyuluh' => 'Petani 1',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu 1',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 1.5,
            'status' => 'belum_terinput'
        ]);

        Petani::create([
            'nama_penyuluh' => 'Petani 2',
            'no_ktp' => '6543210987654321',
            'nama_ibu_kandung' => 'Nama Ibu 2',
            'kode_desa' => 'DESA002',
            'id_poktan' => 'POKTAN002',
            'kode_kios' => 'KIOS002',
            'subsektor' => 'hortikultura',
            'komoditas' => 'Jagung',
            'luas_lahan' => 2.5,
            'status' => 'sudah_terinput'
        ]);

        // Act
        $allPetani = $this->repository->getAll();

        // Assert
        $this->assertCount(2, $allPetani);
        $this->assertEquals('Petani 1', $allPetani[0]->nama_penyuluh);
        $this->assertEquals('Petani 2', $allPetani[1]->nama_penyuluh);
    }
}