<?php

namespace Tests\Unit;

use App\Models\Petani;
use App\Models\User;
use App\Repositories\Interfaces\PetaniRepositoryInterface;
use App\Services\PetaniService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Mockery;
use Tests\TestCase;
use Illuminate\Support\Facades\DB;

class PetaniServiceTest extends TestCase
{
    use RefreshDatabase;

    protected $petaniRepository;
    protected $petaniService;

    public function setUp(): void
    {
        parent::setUp();
        
        // Mock repository
        $this->petaniRepository = Mockery::mock(PetaniRepositoryInterface::class);
        
        // Create service with mocked repository
        $this->petaniService = new PetaniService($this->petaniRepository);
    }

    public function tearDown(): void
    {
        Mockery::close();
        parent::tearDown();
    }

    /** @test */
    public function it_can_create_petani_with_valid_data()
    {
        // Arrange
        $userData = [
            'nama_penyuluh' => 'Test Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 1.5,
            'status' => 'belum_terinput',
            'alamat' => 'Alamat Test',
            'no_hp' => '081234567890',
            'email' => 'test@example.com',
            'jenis_kelamin' => 'L',
            'tanggal_lahir' => '1990-01-01'
        ];

        // Create a user for auth
        $user = User::factory()->create();
        $this->actingAs($user);

        // Mock DB transaction
        DB::shouldReceive('beginTransaction')->once();
        DB::shouldReceive('commit')->once();
        
        // Mock Log
        Log::shouldReceive('info')->once();

        // Mock repository create method
        $petani = new Petani($userData);
        $petani->id_petani = 'TEST001';
        $this->petaniRepository->shouldReceive('create')
            ->once()
            ->with(Mockery::on(function ($data) use ($userData) {
                return $data['nama_penyuluh'] === $userData['nama_penyuluh'] &&
                       $data['no_ktp'] === $userData['no_ktp'];
            }))
            ->andReturn($petani);

        // Act
        $result = $this->petaniService->createPetani($userData);

        // Assert
        $this->assertEquals($petani, $result);
        $this->assertEquals('TEST001', $result->id_petani);
        $this->assertEquals($userData['nama_penyuluh'], $result->nama_penyuluh);
    }

    /** @test */
    public function it_throws_validation_exception_for_invalid_data()
    {
        // Arrange
        $invalidData = [
            'nama_penyuluh' => '', // required field is empty
            'no_ktp' => '12345', // too short
            'subsektor' => 'invalid_value', // invalid enum
        ];

        // Act & Assert
        $this->expectException(ValidationException::class);
        $this->petaniService->createPetani($invalidData);
    }

    /** @test */
    public function it_can_update_petani_with_valid_data()
    {
        // Arrange
        $petaniId = 'TEST001';
        $userData = [
            'nama_penyuluh' => 'Updated Petani',
            'no_ktp' => '1234567890123456',
            'nama_ibu_kandung' => 'Nama Ibu Updated',
            'kode_desa' => 'DESA001',
            'id_poktan' => 'POKTAN001',
            'kode_kios' => 'KIOS001',
            'subsektor' => 'tanaman_pangan',
            'komoditas' => 'Padi',
            'luas_lahan' => 2.0,
            'status' => 'sudah_terinput',
        ];

        // Create a user for auth
        $user = User::factory()->create();
        $this->actingAs($user);

        // Mock DB transaction
        DB::shouldReceive('beginTransaction')->once();
        DB::shouldReceive('commit')->once();
        
        // Mock Log
        Log::shouldReceive('info')->once();

        // Mock repository methods
        $petani = new Petani($userData);
        $petani->id_petani = $petaniId;
        
        $this->petaniRepository->shouldReceive('findById')
            ->once()
            ->with($petaniId)
            ->andReturn($petani);
            
        $this->petaniRepository->shouldReceive('update')
            ->once()
            ->with($petaniId, Mockery::on(function ($data) use ($userData) {
                return $data['nama_penyuluh'] === $userData['nama_penyuluh'] &&
                       $data['luas_lahan'] === $userData['luas_lahan'];
            }))
            ->andReturn($petani);

        // Act
        $result = $this->petaniService->updatePetani($petaniId, $userData);

        // Assert
        $this->assertEquals($petani, $result);
        $this->assertEquals($petaniId, $result->id_petani);
        $this->assertEquals($userData['nama_penyuluh'], $result->nama_penyuluh);
    }

    /** @test */
    public function it_can_delete_petani()
    {
        // Arrange
        $petaniId = 'TEST001';
        
        // Mock repository methods
        $petani = new Petani();
        $petani->id_petani = $petaniId;
        
        $this->petaniRepository->shouldReceive('findById')
            ->once()
            ->with($petaniId)
            ->andReturn($petani);
            
        $this->petaniRepository->shouldReceive('delete')
            ->once()
            ->with($petaniId)
            ->andReturn(true);

        // Act
        $result = $this->petaniService->deletePetani($petaniId);

        // Assert
        $this->assertTrue($result);
    }

    /** @test */
    public function it_throws_exception_when_deleting_nonexistent_petani()
    {
        // Arrange
        $petaniId = 'NONEXISTENT';
        
        // Mock repository methods
        $this->petaniRepository->shouldReceive('findById')
            ->once()
            ->with($petaniId)
            ->andReturn(null);

        // Act & Assert
        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('Petani dengan ID NONEXISTENT tidak ditemukan');
        
        $this->petaniService->deletePetani($petaniId);
    }
}